%% Replication code for: 
% COMMON FACTOR OF COMMODITY PRICES (2021)
% by Simona Delle Chiaie, Laurent Ferrara and Domenico Giannone
% Corresponding author: S. Delle Chiaie 
% (email:simona.dellechiaie@ecb.europa.eu)
% This program has been written for the MATLAB version R2020a
% The software can be freely used in applications. 
% Users are kindly requested to add acknowledgements to published work and 
% to cite the above reference in any resulting publications.

% First version: Paris, 13 April 2015
% last update: Frankfurt am Main, July 2021

clear;
clc;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Section 1. set path and folders
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

rootfolder= pwd;
dataset = strcat(rootfolder,'\dataset\'); % data path
    if ~exist(dataset, 'dir')
    mkdir(dataset);
    end
functions = strcat(rootfolder,'\functions\'); % functions
    if ~exist(functions, 'dir')
    mkdir(functions);
    end
output = strcat(rootfolder,'\output\'); % results
    if ~exist(output, 'dir')
     mkdir(output);
    end
addpath(dataset); addpath(functions); addpath(output); % output

namesave = strcat('sav_',datestr(today)); % date

%% Section 2: Load data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[DATA,TEXT]= xlsread('Commodities_IMF_data.xlsx','data'); 
Time = datenum((TEXT(2:end,1)));
X = DATA;
Mnem = TEXT(1,2:end)';
[DATA,TEXT]= xlsread('Commodities_IMF_data.xlsx','legend');
Unit = TEXT(2:end,2);
Description = TEXT(2:end,3);
BlockNames = TEXT(1,4:end-1)';
Blocks = DATA(:,1:end-1);
Global = DATA(:,1);
Select = DATA(:,end);
[DATA,TEXT]= xlsread('Commodities_IMF_data.xlsx','weights');
Weights = DATA; 
clear DATA TEXT   
disp ('Section 2: Data loaded')


%% Section 3: Create variables and select data for estimation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

X = X(:,Select==1);
Global = Global(Select==1);
Description = Description(Select==1);
Blocks = Blocks(Select==1,:);
Unit = Unit(Select==1);
Mnem = Mnem(Select==1);
Xdfm = X(:,Global==1); 
Xind = X(:,Global==0);
MnemDfm = Mnem(Global==1);
BlockDfm = Blocks(Global==1,:);
DescriptionDfm = Description(Global==1);

x = diff(log(Xdfm))*100;
y = diff(log(Xind))*100;


TimeDfm = Time(2:end,:);
DatesDfm = datevec(TimeDfm);

% select estimation sample
BegEstY = 1981; BegEstM = 1;
EndEstY = 2020; EndEstM = 3; 

TbegEst = find(DatesDfm(:,1)==BegEstY & DatesDfm(:,3)==BegEstM); 
TendEst = find(DatesDfm(:,1)==EndEstY & DatesDfm(:,3)==EndEstM);

xest = x(TbegEst:TendEst,:);
DatesEst = DatesDfm(TbegEst:TendEst,:);
TimeEst = datenum(DatesEst);

[t,m] = size(xest);

xin = xest;

xin(xin==0) = NaN;

% check missing data
SerOK = (sum(isnan(xin))<t/4); 

xin= xin(:,SerOK);
Mnemest = MnemDfm(SerOK,1);
Blockest = BlockDfm(SerOK,:);
w = Weights(SerOK,:);
Descriptionest = DescriptionDfm(SerOK,1);
[t,m] = size(xin);

clear X Time TimeDfm load_data 
disp ('Section 3: Create variables and select data for estimation ends')

%% Section 4: ESTIMATION DFM WITH BLOCKS USING EM ALGORITHM
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

r = 1; % number of common factors
Par.r = ones(size(Blockest,2),1); Par.r(1) = r; %Number of block factors
Par.p = 1; % Lags in the VAR on the factors
Par.nQ = 0; %n. of quarterly variables, it is necessary for the model to understand the positions
Par.blocks = [Blockest];
Par.thresh = 1e-4;
Par.max_iter = 1000;

disp ('Estimation starts')
Res_in = EM_DFM_SS_block_idioQARMA_restrMQ(xin,Par);

clear Select_blocks TbegEst TendEst 

disp ('Section 4: Estimation completed')

%% Section 5:  FIGURE 1 - The global factor
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

F_t = Res_in.F(:,1);
imf_index = Res_in.X_sm*w(:,1);
time=(1981+1/12:1/12:2020+3/12)';  % Time line

figure(1)
plot(time,imf_index,'Color',[0.5 0.5 0.5],'LineWidth',1.5)    
hold on
plot(time,F_t(:,1),'b','LineWidth',1.5)
legend('IMF commodity index','Global Factor','Location','SouthWest')
legend('boxoff')
set(gca,'FontSize',12,'Fontname','Arial');
axis('tight')
ylim([-20 20])

filename = fullfile(output, 'Figure1');
savefig(filename);
print(filename,'-dpng');
clear time

disp ('Section 5: Figure 1 saved')

%% Section 6: Load additional data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[DATA,TEXT]= xlsread('additional_data.xlsx','data');
Time = datenum((TEXT(2:end,1)));
X = DATA(:,1:end);
Mnem = TEXT(1,2:end)';
clear DATA TEXT

% select sample
Dates = datevec(Time);
% 
BegEstY = 1982; BegEstM = 1;
EndEstY = 2020; EndEstM = 2;
% 
TbegEst = find(Dates(:,1)==BegEstY & Dates(:,3)==BegEstM);
TendEst = find(Dates(:,1)==EndEstY & Dates(:,3)==EndEstM);

Y = X(TbegEst:TendEst,:);

clear Text X BegEstY EndEstY TbegEst TendEst

% transform the global factor and select sample
h=12;

y = (filter(ones(1, h) / h, 1, F_t(:,1)))*12;

BegY = 1982; BegM = 1;
EndY = 2020; EndM = 2;
% 
Tbeg = find(DatesEst(:,1)==BegY & DatesEst(:,3)==BegM);
Tend = find(DatesEst(:,1)==EndY & DatesEst(:,3)==EndM);

yplot = y(Tbeg:Tend,1);

clear Da* T* E*

disp ('Section 6: Other data loaded')
% 
%% Section 7:  FIGURE 2 - the global factor and other measures of global economic activity
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

K_t = Y(:,1); % Kilian index (2009,2019)
GF_t = yplot;% global factor
IP_t = Y(:,2);% OECD + 6 IP (as in Baumeister and Hamilton, 2019)

time=(1982+1/12:1/12:2020+2/12)'; 

figure(2)
subplot(2,1,1)
yyaxis left
plot(time,K_t,'LineWidth',1)
yyaxis right
plot(time,GF_t,'LineWidth',1)
ylim([-50 50])
set(gca,'FontSize',12,'Fontname','Arial','xlim',[1982 2020.3]);
legend('Kilian index', 'Global Factor','Location','South');
legend('boxoff');

% NBER recession bars
patch([1982.1 1983.1 1983.1 1982.1],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([1990.7 1991.5 1991.5 1990.7],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([2001.3 2002.10 2002.10 2001.3],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([2007.12 2009.6 2009.6 2007.12],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([2020.1 2020.3 2020.3 2020.1],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');

subplot(2,1,2)
yyaxis left
plot(time,IP_t,'LineWidth',1)
yyaxis right
plot(time,GF_t,'LineWidth',1)
ylim([-50 50])
set(gca,'FontSize',12,'Fontname','Arial','xlim',[1982 2020.3]);
legend('World Industrial Production', 'Global Factor','Location','South');
legend('boxoff')

% NBER recession bars
patch([1982.1 1983.1 1983.1 1982.1],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([1990.7 1991.5 1991.5 1990.7],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([2001.3 2002.10 2002.10 2001.3],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([2007.12 2009.6 2009.6 2007.12],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
patch([2020.1 2020.3 2020.3 2020.1],[-200 -200 200 200],[0.5 0.5 0.5],'Facealpha',0.5,'Linestyle','none', 'HandleVisibility','off');
filename = fullfile(output, 'Figure2');
savefig(filename);
print(filename,'-dpng');

clear IP_t K_t imf_index
disp ('Section 7: Figure 2 saved')

%% Section 8: FIGURE 3 -  Factor loadings
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
C = Res_in.C(:,1).*Res_in.Wx'; 

figure(3)

subplot(2,1,1)
bar(C(1:26,1),0.8), colormap(cool)
ylim([0 3]);
set(gca,'XTick',1:1:size(xin(1:26),2))
set(gca,'XTickLabel',Mnemest(1:26,1),'FontSize',9)
set(gca,'XTickLabelRotation',90)

subplot(2,1,2)
bar(C(27:51,1),0.8), colormap(cool)
ylim([0 3]);
set(gca,'XTick',1:1:size(xin(27:51),2))
set(gca,'XTickLabel',Mnemest(27:end,1),'FontSize',9)
set(gca,'XTickLabelRotation',90)

filename = fullfile(output, 'Figure3');
savefig(filename);
print(filename,'-dpng');

disp ('Section 8: Figure 3 saved')

%% Section 9: Figure 4 Common components

% Copper (9)
% Brent crude oil (21)

a = find(strcmp(Mnemest, 'Copper')); 
b = find(strcmp(Mnemest, 'Brent Crude Oil')); 
%  
Gamma_a = (Res_in.Wx(1,a)).*(Res_in.C(a,1)*F_t(:,1))+Res_in.Mx(1,a);
Gamma_b = (Res_in.Wx(1,b)).*(Res_in.C(b,1)*F_t(:,1))+Res_in.Mx(1,b);

h = 12;

y_c = filter(ones(h,1)/h,1,Gamma_a);
y_o = filter(ones(h,1)/h,1,Gamma_b);
cop_f = filter(ones(h,1)/h,1,Res_in.X_sm(:,a));
oil_f = filter(ones(h,1)/h,1,Res_in.X_sm(:,b));

figure(4)
subplot(2,1,1)
time=(1981+1/12:1/12:2020+3/12)';  

plot(time, oil_f,':','Color',[0 0 0.7],'LineWidth',1)
hold on
plot(time,cop_f,':','Color',[0, 0.6, 0.3],'LineWidth',1)
plot(time,y_o,'Color',[0 0 0.7],'LineWidth',1.5)
plot(time, y_c ,'Color',[0, 0.6,0.3],'LineWidth',1.5)
legend('Copper','Brent Oil','Common component of copper','Common component of oil','Location','SouthEast')
legend('Copper','Brent Oil','Location','SouthEast')
legend BOXOFF
set(gca,'FontSize',12,'Fontname','Arial','xlim',[1981 2020.3]);
hold off

subplot(2,1,2)
plot(time,(oil_f-cop_f),':',time, (y_o-y_c),'LineWidth',1)
legend('Relative prices','Relative common components','Location','SouthEast')
legend BOXOFF
set(gca,'FontSize',12,'Fontname','Arial','xlim',[1981 2020.3]);

clear G*
filename = fullfile(output, 'Figure4');
savefig(filename);
print(filename,'-dpng');

disp ('Section 9: Figure 4 saved')
disp ('Main_est ended')
